<?php
/**
 * Plugin Name: Rosie Website Texting Widget
 * Plugin URI: https://heyrosie.com
 * Description: Adds the Rosie website texting widget to your WordPress site.
 * Version: 1.0.0
 * Author: Rosie
 * Author URI: https://heyrosie.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: rosie-website-texting-widget
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Register the settings page under Settings menu.
 */
function rosie_widget_add_settings_page() {
    add_options_page(
        __( 'Rosie Website Texting Widget', 'rosie-website-texting-widget' ),
        __( 'Rosie', 'rosie-website-texting-widget' ),
        'manage_options',
        'rosie-website-texting-widget',
        'rosie_widget_settings_page'
    );
}
add_action( 'admin_menu', 'rosie_widget_add_settings_page' );

/**
 * Register plugin settings.
 */
function rosie_widget_register_settings() {
    register_setting(
        'rosie_widget_settings',
        'rosie_widget_business_id',
        array(
            'type'              => 'string',
            'sanitize_callback' => 'rosie_widget_sanitize_business_id',
            'default'           => '',
        )
    );
}
add_action( 'admin_init', 'rosie_widget_register_settings' );

/**
 * Sanitize and validate the business ID input.
 *
 * @param string $input The input value.
 * @return string The sanitized value.
 */
function rosie_widget_sanitize_business_id( $input ) {
    return sanitize_text_field( trim( $input ) );
}

/**
 * Render the admin settings page.
 */
function rosie_widget_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    ?>
    <style>
        .rosie-settings-card {
            max-width: 600px;
            background: #fff;
            border: 1px solid #E5E7EB;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            margin-top: 20px;
        }
        .rosie-settings-header {
            background: #9332e0;
            color: #fff;
            padding: 16px 24px;
            border-radius: 8px 8px 0 0;
            font-size: 18px;
            font-weight: 600;
        }
        .rosie-settings-body {
            padding: 24px;
        }
        .rosie-settings-body label {
            display: block;
            font-weight: 600;
            color: #2a143c;
            margin-bottom: 8px;
        }
        .rosie-settings-body input[type="text"] {
            width: 100%;
            padding: 10px 12px;
            border: 1px solid #E5E7EB;
            border-radius: 6px;
            font-size: 14px;
            box-sizing: border-box;
        }
        .rosie-settings-body input[type="text"]:focus {
            outline: none;
            border-color: #9332e0;
            box-shadow: 0 0 0 2px rgba(147, 50, 224, 0.2);
        }
        .rosie-settings-body .description {
            color: #595161;
            font-size: 13px;
            margin-top: 8px;
        }
        .rosie-settings-body .rosie-submit-btn {
            background: #9332e0;
            border: none;
            color: #fff;
            padding: 10px 20px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            margin-top: 16px;
        }
        .rosie-settings-body .rosie-submit-btn:hover {
            background: #7a28bd;
        }
        .rosie-settings-subheader {
            background: #fbfbfb;
            color: #595161;
            padding: 0 24px 16px 24px;
            font-size: 14px;
            line-height: 1.5;
            padding-top: 16px;
            border-bottom: 1px solid #f3f2f5;
        }
        .rosie-settings-subheader a {
            color: #595161 !important;
            text-decoration: underline;
        }
        .rosie-settings-subheader a:hover {
            opacity: 0.8;
        }
    </style>
    <div class="wrap">
        <div class="rosie-settings-card">
            <div class="rosie-settings-header">
                <?php esc_html_e( 'Rosie Website Texting Widget', 'rosie-website-texting-widget' ); ?>
            </div>
            <div class="rosie-settings-subheader">
                <?php
                printf(
                    esc_html__( 'Turn your website visitors into valuable leads with AI-powered text conversations. To update widget settings, go to your %sRosie Admin%s.', 'rosie-website-texting-widget' ),
                    '<a href="https://heyrosie.com/admin/settings;section=website-texting" target="_blank">',
                    '</a>'
                );
                ?>
            </div>
            <div class="rosie-settings-body">
                <form action="options.php" method="post">
                    <?php settings_fields( 'rosie_widget_settings' ); ?>
                    <label for="rosie_widget_business_id">
                        <?php esc_html_e( 'Business ID', 'rosie-website-texting-widget' ); ?>
                    </label>
                    <input
                        type="text"
                        id="rosie_widget_business_id"
                        name="rosie_widget_business_id"
                        value="<?php echo esc_attr( get_option( 'rosie_widget_business_id', '' ) ); ?>"
                    />
                    <p class="description">
                        <?php esc_html_e( 'Find your Business ID in the Installation Code section of your Rosie Admin.', 'rosie-website-texting-widget' ); ?>
                    </p>
                    <button type="submit" class="rosie-submit-btn">
                        <?php esc_html_e( 'Save Changes', 'rosie-website-texting-widget' ); ?>
                    </button>
                </form>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Inject the Rosie Widget script in the footer.
 */
function rosie_widget_inject_script() {
    // Only load on frontend
    if ( is_admin() ) {
        return;
    }

    $business_id = get_option( 'rosie_widget_business_id', '' );

    // Only output if business ID is configured
    if ( empty( $business_id ) ) {
        return;
    }

    printf(
        '<script src="https://widget.heyrosie.com/widget.js" data-rosie-business-id="%s"></script>',
        esc_attr( $business_id )
    );
}
add_action( 'wp_footer', 'rosie_widget_inject_script' );
